import React, { useState } from "react";
import {
    Image,
    ScrollView,
    StyleSheet,
    Text,
    TextInput,
    TouchableOpacity,
    View,
    Keyboard,
    Pressable,
} from "react-native";
import { useDispatch, useSelector } from "react-redux";
import PlusIcon from "../../../assets/svgs/cart/plus.svg";
import TimesIcon from "../../../assets/svgs/drawer/times.svg";
import DeleteIcon from "../../../assets/svgs/my wishlist/delete icon.svg";
import useAuth from "../../../hooks/useAuth";
import { getMyReviews } from "../../../redux/slices/user/myReviews/getMyReviews";
import { postNewReview } from "../../../redux/slices/user/myReviews/postUpdateReview";
import ProgressiveImage from "../../../src/components/ProgressiveImage";
import { writeReviewStyle } from "../../Home/FeaturedProducts/ProductDetails/RatingAndReviews/WriteReviews/WriteReviewsStyle";
import CommonStyles from "../../Utilities/CommonStyles/CommonStyles";
import BackNavigation from "../../Utilities/CustomHeader/BackNavigation";
import CustomSpinner from "../../Utilities/CustomSpinner/CustomSpinner";
import dpr from "../../Utilities/CustomStyleAttribute/dpr";
import { mutation } from "../../../redux/slices/util/apiRequest";
import useUploadMultipleFile from "../../../hooks/useUploadMultipleFile";
import { BASE_API_URL_STG } from "@env";
import CustomRating from "../../Utilities/CustomRating/CustomRating";
import { round } from "../../../helper/ratingRound";
import { InOnScreenLoader } from "../../ShoppingCart/ShoppingCart";
import useCustomToast from "../../../hooks/useCustomToast";
import { useNavigation } from "@react-navigation/native";
import useLangTranslation from "../../../hooks/useLangTranslation";
import CustomSVG from "../../../src/components/CustomSVG/CustomSVG";

const EditReview = (props) => {
    const { trans } = useLangTranslation();
    let maxLength = 500;
    const dispatch = useDispatch();
    const navigation = useNavigation();
    const showToast = useCustomToast();
    const { access_token } = useAuth();
    const [enableUpdate, setEnableUpdate] = useState(false);
    const { loading } = useSelector((state) => state.postNewReview);
    const { item, url } = props.route.params || {};
    const {
        comments,
        created_at,
        product_name,
        rating,
        image,
        id,
        product_image,
    } = item;

    const initialReview = {
        comments: comments || "",
        image: image,
    };
    const [stars, setStars] = useState(rating);
    const [review, setReview] = useState(initialReview);
    const [img, setImg] = useState([]);
    const [removeImage, setRemoveImage] = useState(false);

    const uploadImages = useUploadMultipleFile();

    const getImages = async () => {
        try {
            const result = await uploadImages();
            if (result) {
                setImg([...img, ...result]);
                setEnableUpdate(true);
            }
        } catch (e) {}
    };

    const ratingCompleted = (r) => {
        setStars(round(r));
        setEnableUpdate(true);
    };

    const getComments = (newText) => {
        setReview({ ...review, comments: newText });
        setEnableUpdate(true);
    };

    const handleRemoveImage = async (image) => {
        setRemoveImage(true);
        const DELETE_URL = `${BASE_API_URL_STG}/user/review/delete-file`;
        const data = await mutation(
            DELETE_URL,
            "POST",
            { path: image },
            access_token
        );
        if (data?.response?.status?.code === 200) {
            let restImage = review?.image?.filter((item) => item != image);
            setReview({ ...review, image: restImage });
            setRemoveImage(false);
            showToast({
                text1: trans(data?.response?.records?.message),
                type: "common",
                position: "bottom",
                props: { variant: "success" },
            });
        } else {
            setRemoveImage(false);
            showToast({
                text1: trans("File does not remove. Please try again!"),
                type: "common",
                position: "bottom",
                props: { variant: "error" },
            });
        }
    };
    const handleRemoveImageFromClient = (id) => {
        const restImage = img.filter((_, index) => id != index);
        setImg(restImage);
    };

    const handleReviewUpdate = async (id) => {
        review.rating = stars;
        const updateRevUrl = `${BASE_API_URL_STG}/user/review/update/${id}`;
        let formData = new FormData();
        formData.append("comments", review.comments);
        formData.append("rating", review.rating);
        for (let item of img) {
            formData.append("image[]", item);
        }
        let newReview = await dispatch(
            postNewReview({
                access_token,
                updateRevUrl,
                method: "POST",
                formData,
            })
        );
        let { status, records } = newReview?.payload || {};
        let { code } = status || {};
        let { message } = records || {};

        if (code === 200) {
            showToast({
                text1: trans(message),
                type: "common",
                position: "bottom",
                props: { variant: "success" },
            });
            dispatch(getMyReviews({ access_token, URL: url }));
            props.navigation.navigate("my reviews");
        } else {
            showToast({
                text1: trans(message),
                type: "common",
                position: "bottom",
                props: { variant: "error" },
            });
        }
    };

    const [isDeleteReview, setIsDeleteReview] = useState(false);
    const handleDeleteReview = async () => {
        const deleteReviewUrl = `${BASE_API_URL_STG}/user/review/delete`;
        try {
            setIsDeleteReview(true);
            const res = await mutation(
                deleteReviewUrl,
                "POST",
                { id },
                access_token
            );
            setIsDeleteReview(false);
            dispatch(getMyReviews({ access_token, URL }));
            const { response: { status, records } = {} } = res || {};
            if (status?.code == 200) {
                showToast({
                    text1: trans(records),
                    type: "common",
                    position: "bottom",
                    props: { variant: "success" },
                });
                dispatch(getMyReviews({ access_token, URL: url }));
                navigation.goBack();
            }
        } catch (error) {
            showToast({
                text1: trans("Reviews does not delete"),
                type: "comeditIconmon",
                position: "bottom",
                props: { variant: "error" },
            });
            setIsDeleteReview(false);
        }
    };

    return (
        <View style={CommonStyles.cont}>
            {(removeImage || isDeleteReview) && <InOnScreenLoader />}
            <BackNavigation
                navigationProps={props.navigation}
                routeName={trans("Edit Review")}
                capitalize={true}
            />
            <View
                style={editReviewStyle.editIcon}
            >
                <TouchableOpacity onPress={handleDeleteReview}>
                    <CustomSVG svgIcon={DeleteIcon}
                        height={dpr(19)}
                        width={dpr(19)}
                        fill={"#898989"}
                    />
                </TouchableOpacity>
            </View>
            <ScrollView
                keyboardShouldPersistTaps='always'
                style={CommonStyles.cont}
            >
                <View style={CommonStyles.globalContainer}>
                    <View style={writeReviewStyle.imgRatingCon}>
                        <View style={writeReviewStyle.imgCon}>
                            {product_image && <Image
                                source={{ uri: product_image }}
                                style={writeReviewStyle.productImg}
                            />}
                        </View>
                        <View>
                            <Text style={writeReviewStyle.productName}>
                                {product_name}
                            </Text>
                            <View style={writeReviewStyle.ratingCon}>
                                <CustomRating
                                    rating={round(rating)}
                                    starSize={dpr(18)}
                                    readOnly={true}
                                />
                            </View>
                        </View>
                    </View>
                    <View style={editReviewStyle.reviewDateCon}>
                        <Text style={editReviewStyle.reviewText}>
                            {trans("Reviewed on")}
                        </Text>
                        <Text style={editReviewStyle.reviewText}>
                            {created_at}
                        </Text>
                    </View>
                    <View style={writeReviewStyle.addRatingsCon}>
                        <Text style={writeReviewStyle.addRatingText}>
                            {trans("Your Ratings")}
                        </Text>
                        <View style={writeReviewStyle.addRatingCon}>
                            <View>
                                <CustomRating
                                    rating={round(stars)}
                                    onChange={ratingCompleted}
                                    starSize={dpr(29)}
                                    starStyle={writeReviewStyle.starStyle}
                                />
                            </View>
                            <View>
                                <Text style={writeReviewStyle.ratingText}>
                                    {stars} {trans("Stars")}
                                </Text>
                            </View>
                        </View>
                    </View>
                    <View style={writeReviewStyle.addRatingsCon}>
                        <Text style={writeReviewStyle.addRatingText}>
                            {trans("Your Photos")}
                        </Text>

                        <View style={editReviewStyle.imgContainer}>
                            {review?.image?.length > 0 &&
                                review?.image?.map((img, i) => {
                                    return (
                                        <View key={i}>
                                            <ProgressiveImage
                                                source={{ uri: img }}
                                                style={editReviewStyle.imgStyle}
                                            />
                                            <TouchableOpacity
                                                onPress={() =>
                                                    handleRemoveImage(img)
                                                }
                                                style={editReviewStyle.timesIconButton}
                                            >
                                                <CustomSVG svgIcon={TimesIcon}
                                                    fill='#2c2c2c'
                                                    width={dpr(11)}
                                                    height={dpr(11)}
                                                />
                                            </TouchableOpacity>
                                        </View>
                                    );
                                })}
                            {img.map((image, i) => {
                                return (
                                    <View key={i}>
                                        {image?.uri && <Image
                                            source={{ uri: image?.uri }}
                                            style={editReviewStyle.imgStyle}
                                        />}
                                        <TouchableOpacity
                                            onPress={() =>
                                                handleRemoveImageFromClient(i)
                                            }
                                            style={editReviewStyle.handleRemoveImageFrom}
                                        >
                                            <CustomSVG svgIcon={TimesIcon}
                                                fill='#2c2c2c'
                                                width={dpr(11)}
                                                height={dpr(11)}
                                            />
                                        </TouchableOpacity>
                                    </View>
                                );
                            })}
                            <TouchableOpacity onPress={getImages}>
                                <View style={editReviewStyle.addPhotosCon}>
                                    <CustomSVG svgIcon={PlusIcon}
                                        height={dpr(16)}
                                        width={dpr(16)}
                                        fill={"#c4c4c4"}
                                    />
                                </View>
                            </TouchableOpacity>
                        </View>
                    </View>

                    <View style={writeReviewStyle.addRatingsCon}>
                        <Text style={writeReviewStyle.addRatingText}>
                            {trans("Your Experience")}
                        </Text>
                        <View>
                            <TextInput
                                style={writeReviewStyle.textInputStyle}
                                multiline
                                numberOfLines={8}
                                maxLength={maxLength}
                                onChangeText={getComments}
                                defaultValue={comments}
                                onSubmitEditing={Keyboard.dismiss}
                            ></TextInput>
                        </View>
                        <Text style={writeReviewStyle.remainingChar}>
                            {trans("{{x}} characters remaining", {x: maxLength - review?.comments?.length})}
                        </Text>
                    </View>
                </View>
            </ScrollView>
            <Pressable
                disabled={enableUpdate ? false : true}
                onPress={!loading ? () => handleReviewUpdate(id) : null}
            >
                <View style={writeReviewStyle.submitReview}>
                    <View
                        style={[
                            writeReviewStyle.submitReviewCon,
                            writeReviewStyle.bg(enableUpdate)
                        ]}
                    >
                        {enableUpdate && loading ? (
                            <CustomSpinner
                                filePath={require("../../../assets/lottie/loader2.json")}
                                size={{
                                    width: dpr(60),
                                    height: dpr(55),
                                }}
                            />
                        ) : (
                            <Text
                                style={[
                                    writeReviewStyle.submitReviewText,
                                    writeReviewStyle.color(enableUpdate)
                                ]}
                            >
                                {trans("Update Review")}
                            </Text>
                        )}
                    </View>
                </View>
            </Pressable>
        </View>
    );
};

export default EditReview;

export const editReviewStyle = StyleSheet.create({
    reviewDateCon: {
        marginTop: dpr(20),
        flexDirection: "row",
        justifyContent: "space-between",
    },
    reviewText: {
        fontFamily: "Roboto_500Medium",
        fontSize: dpr(14),
        color: "#898989",
    },
    imgContainer: {
        flexDirection: "row",
        flexWrap: "wrap",
    },
    imgStyle: {
        height: dpr(56),
        width: dpr(56),
        borderRadius: 4,
        marginRight: dpr(12),
        marginBottom: dpr(12),
        position: "relative",
    },
    addPhotosCon: {
        backgroundColor: "#F3F3F3",
        height: dpr(56),
        width: dpr(56),
        borderRadius: 4,
        borderStyle: "dashed",
        borderColor: "#898989",
        borderWidth: 1,
        justifyContent: "center",
        alignItems: "center",
    },
    handleRemoveImageFrom: {
        backgroundColor: "#FCCA19",
        position: "absolute",
        bottom: dpr(54),
        right: dpr(7),
        borderRadius: dpr(10),
        padding: dpr(5),
    },
    editIcon: {
        position: "absolute",
        right: dpr(20),
        top: dpr(26),
    },
    timesIconButton: {
        backgroundColor: "#FCCA19",
        position: "absolute",
        bottom: dpr(54),
        right: dpr(7),
        borderRadius: dpr(10),
        padding: dpr(5),
    }
});
